      FUNCTION V(DR)
!Returns the value of the electronic energy E_el(R) (Eq. 12) in
!J. Komasa, M. Puchalski, P. Czachorowski, G. Lach, K. Pachucki, Phys. Rev. A 100, 032519 (2019)
!for a given internuclear distance "DR", serving as
!a potential in the nuclear radial Schrodinger equation.
!Analytic fit by G. Lach, based on K. Pachucki, Phys. Rev. A 82, 032509 (2010)
      IMPLICIT NONE
      REAL*16 R,EHe,C(6:30),B(1:15),bb,A(1:22),T1,T2,T3,SR,F,BR
      real*16 V,DR
      INTEGER I

      R = DR
      SR = SQRT(R)
      EHe=-2.903724377034119Q0
      bb=1.84024290611525Q0
      BR = bb*R

      C(6)=6.4990267054058393131Q0
      C(7)=0.Q0
      C(8)=124.3990835836223436Q0
      C(9)=0.Q0
      C(10)=3285.828414967421697Q0
      C(11)=-3474.898037882Q0
      C(12)=122727.6087007Q0
      C(13)=-326986.9240441Q0
      C(14)=6361736.045092Q0
      C(15)=-28395580.63300Q0
      C(16)=441205192.2739Q0
      C(17)=-2739281653.140Q0
      C(18)=39352477334.60Q0
      C(19)=-307082459389.3Q0
      C(20)=4363762779418.Q0
      C(21)=-40360383996600.Q0
      C(22)=586030516404600.Q0
      C(23)=-6200210182655000.Q0
      C(24)=93433616652420000.Q0
      C(25)=-1105388541580000000.Q0
      C(26)=17414122541340000000.Q0
      C(27)=-226839565464200000000.Q0
      C(28)=3747403197985000000000.Q0
      C(29)=-53148855088090000000000.Q0
      C(30)=921579197325600000000000.Q0

      B(1)=-4.57323700588624794726213559410563Q0
      B(2)=13.47638719366600967683673663479999Q0
      B(3)=-28.51530241703605266394033745126115Q0
      B(4)=45.78475015811747771910888974954924Q0
      B(5)=-58.19615893167528525478814306565867Q0
      B(6)=58.48593413367164587893380279407919Q0
      B(7)=-45.35039381018086925859899857147845Q0
      B(8)=26.47049312447772391118586407437590Q0
      B(9)=-11.37752030083640364078327219266986Q0
      B(10)=3.50402820299159351290056244127085Q0
      B(11)=-0.73796135013579258240464270522277Q0
      B(12)=0.09631412647157079742433223552984Q0
      B(13)=-0.00576418363156423769040137255530Q0
      B(14)=-0.00014088867788010427947798439255Q0
      B(15)=0.00002845940482698706139384277587Q0

      A(1)=B(1)
      A(2)=B(2)+2+(EHe+1)
      A(3)=B(3)+A(2)*B(1)-B(1)*B(2)
      A(4)=B(4)+2*(A(2)-B(2)-1)+B(2)*(A(2)-B(2))
      A(5)=B(5)+2*B(1)*(A(2)-B(2)-1)+B(3)*(A(2)-B(2))
      A(6)=39.91391743339803899150896228191386Q0
      A(7)=-5.75063757195175050513832273504327Q0
      A(8)=-33.46588381771842371982158706320471Q0
      A(9)=55.32089352568815354527823571430595Q0
      A(10)=-51.42284107211464070094798429642712Q0
      A(11)=32.71776925369248112208919743003516Q0
      A(12)=-15.66016122411670563080887658243106Q0
      A(13)=7.50377730360456765232166233248396Q0
      A(14)=-5.40306648644156257500314013983694Q0
      A(15)=4.47594305162345652142325458988993Q0
      A(16)=-2.95983991352400754745996259257977Q0
      A(17)=1.42669802756450786769491630785461Q0
      A(18)=-0.49306757735153614794116509830620Q0
      A(19)=0.11952677096537517464084697055123Q0
      A(20)=-0.01932309354268259675592903647346Q0
      A(21)=0.00186812017042699683819835094121Q0
      A(22)=-0.00008146220049329661292603862566Q0

      T1=0.Q0
      DO I=1,22
         T1 = T1+A(I)*SR**I
      ENDDO

      T2=0.Q0
      DO I=1,15
         T2 = T2+B(I)*SR**I
      ENDDO

      T3=0.Q0
      DO I=6,30
         T3 = T3-F(I+1,BR)*C(I)/R**I
      ENDDO
      V=(1+T1)/(1+T2)*Exp(-2*R)/R+T3

      RETURN
      END

!----------------------------------------------------------------------
      FUNCTION F(N,X)
!Tang-Toennies damping function.
!K. T. Tang, J. P. Toennies, J. Chem. Phys. 80, 3726 (1984)
!Needed to remove the divergent behavior
!of the 1/X^N factors in the fit near X=0.
!F=1-GAMMA(1+N,X)/GAMMA(1+N)
      IMPLICIT NONE
      REAL*16 F,X,T1,T2
      INTEGER N,I
      INTEGER,PARAMETER :: MAXI=32

      IF(X.GE.3.Q0) THEN
         T1=0.Q0
         T2=1.Q0
         DO I=1,N+1
            T1 = T1+T2
            T2 = T2*X/I
         ENDDO
         F = 1-Exp(-X)*T1
      ELSE
         T1=0
         T2=1
         DO I=0,MAXI
            T1 = T1+X**I/(T2*(N+1+I))
            T2 = -T2*(I+1)
         ENDDO
         T2=1
         DO I=1,N
            T2=T2*I
         ENDDO
         F=X**(N+1)*T1/T2
      ENDIF

      RETURN
      END

!----------------------------------------------------------------------

      real*16 function Eaint(R)
!Adiabatic correction to the potential 2*mu_n*E_a(R).
!See Eq. 18 and the text below in 
!J. Komasa, M. Puchalski, P. Czachorowski, G. Lach, K. Pachucki, Phys. Rev. A 100, 032519 (2019)
!Taken from K. Pachucki and J. Komasa, J. Chem. Phys. 141, 224103 (2014) (supplement)
      implicit none
      real*16 :: R,W1,W2,a,b,c,Eaint0
      real*16 :: mpPJ12,A6,A8,A10,Rinvs
      real*16, dimension(:) :: P(0:14),Q(32)
          a=1.1114055293534456784057320121522Q0
          b=1.2095560917705853428726384695475Q0
          c=1.01490515805955384474660168480738Q0
          Eaint0=0.5313969260599797962744365Q0
          P(0)=2.57640985968790283123151781491906Q4
          P(1)=-4.24766902765405008898541826259253Q4
          P(2)=3.23914101790994120219160120634889Q4
          P(3)=-1.51487736617849601347139327355656Q4
          P(4)=4.85513784701655588907660942038169Q3
          P(5)=-1.12817968586590251891697217394290Q3
          P(6)=1.96022115958782784119305229987615Q2
          P(7)=-2.58727150904267703054234804239174Q1
          P(8)=2.60684632154178626761090277540466Q0
          P(9)=-1.99532537310733713612913792065941Q-1
          P(10)=1.14207904980296991756950227054134Q-2
          P(11)=-4.74017092281636746493356165838112Q-4
          P(12)=1.34857652007195080104183709321439Q-5
          P(13)=-2.35400021232008399283943817140989Q-7
          P(14)=1.90202486167843259334945031635423Q-9
          Q(1)=3.99485699616875103429516544309691Q4
          Q(2)=-5.44880006664402627114573753359117Q4
          Q(3)=5.26679590123087863666126645727291Q4
          Q(4)=-4.53948569025655989594911809433954Q4
          Q(5)=3.33135367104566309045409601884433Q4
          Q(6)=-2.25922524785635131591735975775914Q4
          Q(7)=1.56343244224922385234291816278842Q4
          Q(8)=-1.60616168851563262719458549830300Q4
          Q(9)=2.73228697168824991946522297552660Q4
          Q(10)=-5.16780392742578518472987095005810Q4
          Q(11)=8.66782887179522293314013939749623Q4
          Q(12)=-1.23395802058510929643696213689536Q5
          Q(13)=1.49032029889160972833144538424413Q5
          Q(14)=-1.53587670024503730585958769007304Q5
          Q(15)=1.35761994668913273821118458600299Q5
          Q(16)=-1.03311683808839637492445134792404Q5
          Q(17)=6.78372389519334445654343720586628Q4
          Q(18)=-3.84767394910856453874838129483377Q4
          Q(19)=1.88499722967195307004803452140572Q4
          Q(20)=-7.96690860669712153936199181841002Q3
          Q(21)=2.89804982597848873059044487914495Q3
          Q(22)=-9.03966166931006732358453963336031Q2
          Q(23)=2.40513643411929918936145610873524Q2
          Q(24)=-5.41915909139083735974492686116770Q1
          Q(25)=1.02397672252399842815397491089614Q1
          Q(26)=-1.60129916283328421242788004210142Q0
          Q(27)=2.03505145495990689451837401340804Q-1
          Q(28)=-2.04824339024431528193748270566111Q-2
          Q(29)=1.57093891708355488534614263836794Q-3
          Q(30)=-8.62727960846911255035254246400885Q-5
          Q(31)=3.02264161462310781046446491735097Q-6
          Q(32)=-5.07959453408875288585515001119814Q-8
!      data proton=1836.15267377Q0/
      data mpPJ12/1836.15267247Q0/,A6/1.7699Q-2/,A8/0.144Q0/,A10/2.28Q0/
      if(R<=11.5q0) then
        W1=P(0)+R*(P(1)+R*(P(2)+R*(P(3)+R*(P(4)+R*(P(5)+R*(P(6)+&
          R*(P(7)+R*(P(8)+R*(P(9)+R*(P(10)+R*(P(11)+R*(P(12)+&
          R*(P(13)+R*P(14))))))))))))))
        W2=Eaint0-P(0)+R*(Q(1)+R*(Q(2)+R*(Q(3)+R*(Q(4)+R*(Q(5)+&
          R*(Q(6)+R*(Q(7)+R*(Q(8)+R*(Q(9)+R*(Q(10)+R*(Q(11)+&
          R*(Q(12)+R*(Q(13)+R*(Q(14)+R*(Q(15)+R*(Q(16)+R*(Q(17)+&
          R*(Q(18)+R*(Q(19)+R*(Q(20)+R*(Q(21)+R*(Q(22)+R*(Q(23)+&
          R*(Q(24)+R*(Q(25)+R*(Q(26)+R*(Q(27)+R*(Q(28)+R*(Q(29)+&
          R*(Q(30)+R*(Q(31)+R*Q(32))))))))))))))))))))))))))))))))
        Eaint=EXP(-a*R)*W1+EXP(-(b+c*R)*R)*W2
      else
        Rinvs=1.Q0/(R*R)
        Eaint=-mpPJ12*(A6+(A8+A10*Rinvs)*Rinvs)*Rinvs*Rinvs*Rinvs
      endif
      return
      end function Eaint

!----------------------------------------------------------------------

      real*16 function WTint(R)
!Fit evaluating 'W(R) perpendicular' for 'perpendicular mass', mu_n^2*(W_T(R)-W_T(infty)).
!See Eq. 29 in J. Komasa, M. Puchalski, P. Czachorowski, G. Lach, K. Pachucki, Phys. Rev. A 100, 032519 (2019).
!Taken from K. Pachucki and J. Komasa, J. Chem. Phys. 143, 034111 (2015)  (supplement)
      implicit none
      real*16,intent(in):: R
      real*16 :: W1,W2,a,b,c,R0y,iR,iRs
      real*16, dimension(:) :: P(0:16),Q(20),Ay(8:16)
      R0y=11.700Q0
      a=1.10466168412365114082230970833773Q0
      b=1.161434318812792614926969387240357Q0
      c=1.046019881236837656855107076466617Q0
      P(0)=0.2500000000000000000000000000000000Q0
      P(1)=6.788368024999779419913194963008291Q2
      P(2)=-1.362735040680007543440906343074785Q3
      P(3)=1.248183035099494960499932556222404Q3
      P(4)=-6.944060098081972034423390892296834Q2
      P(5)=2.635915587620845286961548226369195Q2
      P(6)=-7.255938965457191816126997779995122Q1
      P(7)=1.499768830086866823227587156124188Q1
      P(8)=-2.373924273015117946724345451665213Q0
      P(9)=2.904265411458847153936538485775932Q-1
      P(10)=-2.748163855210913108514732051877217Q-2
      P(11)=1.995898365262805614718070205201716Q-3
      P(12)=-1.092949629575136576748615003661843Q-4
      P(13)=4.369218104334472485569100615549883Q-6
      P(14)=-1.203970910005450287181067618972663Q-7
      P(15)=2.045074739418371464155512607549481Q-9
      P(16)=-1.614352339135951222268726319239260Q-11
      Q(1)=-6.785606351132608284173242273496553Q2
      Q(2)=1.324342492554007557452444310749662Q3
      Q(3)=-1.881657496129101928959697388905893Q3
      Q(4)=2.011052281533817927014938952760115Q3
      Q(5)=-1.823196186854297679958098517952106Q3
      Q(6)=1.437634373727193894011885147447552Q3
      Q(7)=-1.017709137778967053385131828632438Q3
      Q(8)=6.559309168188208408363840776134244Q2
      Q(9)=-3.887830008008903443213694021396870Q2
      Q(10)=2.122975068482467459591181482029274Q2
      Q(11)=-1.061259419258636204725815501953624Q2
      Q(12)=4.787640235732326507937109216579178Q1
      Q(13)=-1.907668662497768357006933804047277Q1
      Q(14)=6.540681471331710194291223023450490Q0
      Q(15)=-1.871857575215398239010331068557770Q0
      Q(16)=4.316507935982159667864806144908653Q-1
      Q(17)=-7.664370022485299008451025592197587Q-2
      Q(18)=9.812363927280630019869065080607696Q-3
      Q(19)=-8.060701137136874317641474179597920Q-4
      Q(20)=3.216689269783762022735043976067475Q-5

      Ay(8)=4.435688079888515819759877021567017Q2
      Ay(9)=5.843631859649975894130889399942992Q2
      Ay(10)=2.855935720592942981839687294786613Q3
      Ay(11)=1.034785873697323483958482380157638Q4
      Ay(12)=1.347512954099471093223216915047132Q4
      Ay(13)=1.366010523117704542445979550539646Q4
      Ay(14)=-5.457728060433510988770682809103463Q9
      Ay(15)=1.020758850358781126515006597867726Q11
      Ay(16)=-4.305280665734900196739551711572737Q11
      if(R<=R0y) then
        W1=P(0)+R*(P(1)+R*(P(2)+R*(P(3)+R*(P(4)+R*(P(5)+R*(P(6)+&
          R*(P(7)+R*(P(8)+R*(P(9)+R*(P(10)+R*(P(11)+R*(P(12)+&
          R*(P(13)+R*(P(14)+R*(P(15)+R*P(16))))))))))))))))
        W2=R*(Q(1)+R*(Q(2)+R*(Q(3)+R*(Q(4)+R*(Q(5)+&
          R*(Q(6)+R*(Q(7)+R*(Q(8)+R*(Q(9)+R*(Q(10)+R*(Q(11)+&
          R*(Q(12)+R*(Q(13)+R*(Q(14)+R*(Q(15)+R*(Q(16)+R*(Q(17)+&
          R*(Q(18)+R*(Q(19)+R*Q(20))))))))))))))))))))
        WTint=EXP(-a*R)*W1+EXP(-(b+c*R)*R)*W2
      else
        iR=1.Q0/R
        iRs=iR*iR
        WTint=iRs*iRs*iRs*iRs*(Ay(8)+iR*(Ay(9)+iR*(Ay(10)+iR*(Ay(11)+&
         iR*(Ay(12)+iR*(Ay(13)+iR*(Ay(14)+iR*(Ay(15)+iR*Ay(16)))))))))
      endif
      return
      end function WTint

!----------------------------------------------------------------------

      subroutine WIIint(R,WII,DWII,D2WII)
!Evaluates 'W(R) parallel' for 'parallel mass', mu_n^2*(W_II(R)-W_II(infty))
!and its two derivatives over R.
!See Eq. 28 in J. Komasa, M. Puchalski, P. Czachorowski, G. Lach, K. Pachucki, Phys. Rev. A 100, 032519 (2019).
!Based on K. Pachucki and J. Komasa, J. Chem. Phys. 143, 034111 (2015) (supplement)
      implicit none
      real*16,intent(in)::R
      double precision,intent(out)::WII,DWII,D2WII
      real*16 :: W1,W2,a,b,c,R0x,iR,iRs,ar,ar2,br,br2,cr2
      real*16, dimension(:) :: P(0:14),Q(16),Ax(8:16)
      R0x=10.735Q0
      a=1.027993101600837737184971037577331Q0
      b=1.320414476926467805709407411384579Q0
      c=1.029005093533914595830502921046611Q0
      P(0)=+0.2500000000000000000000000000000000Q0
      P(1)=-6.976823128243076583850090191393916Q3
      P(2)=1.241480671208678824828299918643230Q4
      P(3)=-1.001751495262024425392304467319746Q4
      P(4)=4.867064368980094428678084044858411Q3
      P(5)=-1.592489422549091326324630926192373Q3
      P(6)=3.712095393409728282013822295451242Q2
      P(7)=-6.349792893773727584013276012820269Q1
      P(8)=8.077184560768227216832903374994982Q0
      P(9)=-7.644986831010505426138229926574211Q-1
      P(10)=5.319812137594396302595713165027699Q-2
      P(11)=-2.646159974926613391981967790859699Q-3
      P(12)=8.911882451263913749641066169101843Q-5
      P(13)=-1.821723698506568722000371744995044Q-6
      P(14)=1.707271641467195828769878041215292Q-8
      Q(1)=6.977081281595911959289723020813841Q3
      Q(2)=-1.038053012823322092759820669980730Q4
      Q(3)=1.389481919759480026346938674264039Q4
      Q(4)=-1.321628679624986197978310233015459Q4
      Q(5)=1.137027664267665553883759599472471Q4
      Q(6)=-8.546490000464176257553253450687466Q3
      Q(7)=6.054218225082539541960429281229023Q3
      Q(8)=-4.009144503562168745107926746173330Q3
      Q(9)=2.472328460627753527743876759176205Q3
      Q(10)=-1.354667138256088687480506101755758Q3
      Q(11)=6.283112901110511363001303558533226Q2
      Q(12)=-2.338932767455568399739974166698976Q2
      Q(13)=6.629522652927474404413079489175476Q1
      Q(14)=-1.336480474379359139272781358684217Q1
      Q(15)=1.711285558088694318564815232677170Q0
      Q(16)=-1.062557417129759015766001501312952Q-1

      Ax(8)=-2.775446458388230741646178819172860Q3
      Ax(9)=-7.951892799385567386177310989166826Q2
      Ax(10)=-4.137230766688273651002871019225784Q3
      Ax(11)=-1.213398391622305653582555414444572Q4
      Ax(12)=-1.522888373549849834787939207748314Q4
      Ax(13)=-1.543890538400735949471433035921675Q4
      Ax(14)=1.501821349134892445485788447189599Q10
      Ax(15)=-4.048413796310053899858315044534921Q11
      Ax(16)=1.996992470864601350253850629744593Q12

      if(R<=R0x) then
        W1=P(0)+R*(P(1)+R*(P(2)+R*(P(3)+R*(P(4)+R*(P(5)+R*(P(6)+&
          R*(P(7)+R*(P(8)+R*(P(9)+R*(P(10)+R*(P(11)+R*(P(12)+&
          R*(P(13)+R*(P(14)))))))))))))))
        W2=R*(Q(1)+R*(Q(2)+R*(Q(3)+R*(Q(4)+R*(Q(5)+&
          R*(Q(6)+R*(Q(7)+R*(Q(8)+R*(Q(9)+R*(Q(10)+R*(Q(11)+&
          R*(Q(12)+R*(Q(13)+R*(Q(14)+R*(Q(15)+R*(Q(16)&
          ))))))))))))))))
        WII=EXP(-a*R)*W1+EXP(-(b+c*R)*R)*W2

        W1=-a*P(0)+P(1)+R*(-a*P(1)+2*P(2)+R*(-a*P(2)+3*P(3)+&
          R*(-a*P(3)+4*P(4)+R*(-a*P(4)+5*P(5)+R*(-a*P(5)+6*P(6)+&
          R*(-a*P(6)+7*P(7)+R*(-a*P(7)+8*P(8)+R*(-a*P(8)+9*P(9)+&
          R*(-a*P(9)+10*P(10)+R*(-a*P(10)+11*P(11)+&
          R*(-a*P(11)+12*P(12)+R*(-a*P(12)+13*P(13)+&
          R*(-a*P(13)+14*P(14)-a*R*P(14))))))))))))))
        W2=Q(1)+R*(-(b*Q(1))+2*Q(2)+R*(-2*c*Q(1)-b*Q(2)+3*Q(3)+&
          R*(-2*c*Q(2)-b*Q(3)+4*Q(4)+R*(-2*c*Q(3)-b*Q(4)+5*Q(5)+&
          R*(-2*c*Q(4)-b*Q(5)+6*Q(6)+R*(-2*c*Q(5)-b*Q(6)+7*Q(7)+&
          R*(-2*c*Q(6)-b*Q(7)+8*Q(8)+R*(-2*c*Q(7)-b*Q(8)+9*Q(9)+&
          R*(-2*c*Q(8)-b*Q(9)+10*Q(10)+&
          R*(-2*c*Q(9)-b*Q(10)+11*Q(11)+&
          R*(-2*c*Q(10)-b*Q(11)+12*Q(12)+&
          R*(-2*c*Q(11)-b*Q(12)+13*Q(13)+&
          R*(-2*c*Q(12)-b*Q(13)+14*Q(14)+&
          R*(-2*c*Q(13)-b*Q(14)+15*Q(15)+&
          R*(-2*c*Q(14)-b*Q(15)+16*Q(16)+&
          R*(-2*c*Q(15)-b*Q(16)-2*c*R*Q(16)))))))))))))))))
        DWII=EXP(-a*R)*W1+EXP(-(b+c*R)*R)*W2

        ar=a*r
        ar2=ar**2
        br=b*r
        br2=b**2*r**2+4*b*c*r**3+4*c**2*r**4
        cr2=c*r**2
        W1= a**2*P(0) + (-2*a + a**2*r)*P(1) + &
            (2 - 4*ar + ar2)*P(2) +&
            r*((6 - 6*ar + ar2)*P(3) + &
            r*((12 - 8*ar + ar2)*P(4) +& 
            r*((20 - 10*ar + ar2)*P(5) +& 
            r*((30 - 12*ar + ar2)*P(6) +& 
            r*((42 - 14*ar + ar2)*P(7) +& 
            r*((56 - 16*ar + ar2)*P(8) +& 
            r*((72 - 18*ar + ar2)*P(9) +& 
            r*((90 - 20*ar + ar2)*P(10) +& 
            r*((110 - 22*ar + ar2)*P(11) +& 
            r*((132 - 24*ar + ar2)*P(12) +& 
            r*((156 - 26*ar + ar2)*P(13) +& 
            r*(182 - 28*ar + ar2)*P(14))))))))))))


        W2= (-2*b + b**2*r - 6*c*r + 4*c**2*r**3 + 4*c*b*r**2)*q(1) +& 
            (2 - 4*br + br2 - 10*cr2)*q(2) + &
            r*((6 - 6*br + br2 - 14*cr2)*q(3) +& 
            r*((12 - 8*br + br2 - 18*cr2)*q(4) +& 
            r*((20 - 10*br + br2 - 22*cr2)*q(5) +& 
            r*((30 - 12*br + br2 - 26*cr2)*q(6) + &
            r*((42 - 14*br + br2 - 30*cr2)*q(7) + &
            r*((56 - 16*br + br2 - 34*cr2)*q(8) + &
            r*((72 - 18*br + br2 - 38*cr2)*q(9) + &
            r*((90 - 20*br + br2 - 42*cr2)*q(10) + &
            r*((110 - 22*br + br2 - 46*cr2)*q(11) + &
            r*((132 - 24*br + br2 - 50*cr2)*q(12) + &
            r*((156 - 26*br + br2 - 54*cr2)*q(13) + &
            r*((182 - 28*br + br2 - 58*cr2)*q(14) + &
            r*((210 - 30*br + br2 - 62*cr2)*q(15) + &
            r*(240 - 32*br + br2 - 66*cr2)*q(16))))))))))))))
        D2WII=EXP(-a*R)*W1+EXP(-(b+c*R)*R)*W2
      else
        iR=1.Q0/R
        iRs=iR*iR
        WII=iRs*iRs*iRs*iRs*(Ax(8)+iR*(Ax(9)+iR*(Ax(10)+iR*(Ax(11)+&
         iR*(Ax(12)+iR*(Ax(13)+iR*(Ax(14)+iR*(Ax(15)+iR*Ax(16)))))))))
        DWII=-iR*iRs*iRs*iRs*iRs*(8*Ax(8)+iR*(9*Ax(9)+iR*(10*Ax(10)+&
         iR*(11*Ax(11)+iR*(12*Ax(12)+iR*(13*Ax(13)+iR*(14*Ax(14)+&
         iR*(15*Ax(15)+16*iR*Ax(16)))))))))
       D2WII=iR*iR*iRs*iRs*iRs*iRs*(9*8*Ax(8)+iR*(10*9*Ax(9)+iR*(11*10*Ax(10)+&
         iR*(12*11*Ax(11)+iR*(13*12*Ax(12)+iR*(14*13*Ax(13)+iR*(15*14*Ax(14)+&
         iR*(16*15*Ax(15)+17*16*iR*Ax(16)))))))))

      endif
      return
      end subroutine WIIint

      real*16 function VEna(R)
!Evaluates the total homonuclear nonadiabatic correction to
!the potential: dEna(R)*mu_n**2 from Eq. 34 in 
!J. Komasa, M. Puchalski, P. Czachorowski, G. Lach, K. Pachucki, Phys. Rev. A 100, 032519 (2019).
      implicit none
      real*16,intent(in)::R
      real*16::RU,PV,RPdV
      VEna=RU(R)/R+RPdV(R)/R-PV(R)
      end function

!----------------------------------------------------------------------
      real*16 function PV(R)
!Evaluates  -( 2/R + d/dR )*V(R)*mu_n**2.
!See Eq. 34 J. Komasa, M. Puchalski, P. Czachorowski, G. Lach, K. Pachucki, Phys. Rev. A 100, 032519 (2019).
!From K. Pachucki and J. Komasa, J. Chem. Phys. 143, 034111 (2015) (supplement)
      implicit none
      real*16 :: R,W1,W2,a,b,c,R0v,iR,iRs
      real*16, dimension(:) :: P(1:10),Q(25),Nv(7:17)
          R0v=11.400Q0
          a=1.535295394622434407038306511976158Q0
          b=1.939237383802406686420696985749654Q0
          c=0.8765393769779657938146244104976943Q0
          P(1)=-2.229105452174879771241506337820855Q3
          P(2)=2.177146119178830017743827126096766Q3
          P(3)=-9.553625796629805738288009697781634Q2
          P(4)=2.488551882374273402792672675763945Q2
          P(5)=-4.232811667591062236865870158214063Q1
          P(6)=4.872680670285068219259963723867540Q0
          P(7)=-3.797083023260985267226591352529952Q-1
          P(8)=1.932814893547585415394859049900694Q-2
          P(9)=-5.838657305905051339619769130204564Q-4
          P(10)=8.000104493106888440207458111537249Q-6
          Q(1)=2.219102058527390719662559787486936Q3
          Q(2)=-1.228539933244788338817382389580025Q3
          Q(3)=2.053905309659658279392363272818492Q3
          Q(4)=-8.501959698562834560392697480990884Q2
          Q(5)=1.153047462822826022754661313791882Q3
          Q(6)=-3.092369329038301498789937667040995Q3
          Q(7)=1.086160657133622959283237749539419Q4
          Q(8)=-2.758516425114171499650163126240050Q4
          Q(9)=5.379803897113368729628287616917377Q4
          Q(10)=-8.314259075666691684305936699578876Q4
          Q(11)=1.041051864492128472324934609409652Q5
          Q(12)=-1.068921623669476791887299902472509Q5
          Q(13)=9.062650328618885514558201100716409Q4
          Q(14)=-6.365478117340681595919020702430721Q4
          Q(15)=3.706026323568496089043516437663041Q4
          Q(16)=-1.784948209119919478692012813994252Q4
          Q(17)=7.079275921760616833130636323525926Q3
          Q(18)=-2.294518160014745361230687143552960Q3
          Q(19)=6.008479675458337701008171870551978Q2
          Q(20)=-1.250123861564588553044152330523974Q2
          Q(21)=2.016489726821822964271732069575560Q1
          Q(22)=-2.429165614277015414482755756046675Q0
          Q(23)=2.055799061974127223523851288183537Q-1
          Q(24)=-1.090315952423371576760645347202482Q-2
          Q(25)=2.729046893210881872639213654058394Q-4

          Nv(7)=-7.472182936228868864176663693155737Q1
          Nv(8)=-1.086187434856962296669472219455426Q2
          Nv(9)=-8.564184097290760947811608186492616Q2
          Nv(10)=-2.275930304347636438104799467151248Q3
          Nv(11)=-7.002167474330146679697575802381397Q3
          Nv(12)=-1.245772796290107692168423911595466Q4
          Nv(13)=-1.445995915679935958849996775372432Q4
          Nv(14)=-1.596433835564711100482213306910872Q4
          Nv(15)=4.619370774540983723296448403315812Q10
          Nv(16)=-2.790882334099268464659061917676744Q12
          Nv(17)=1.736331993213003454861657041548325Q13
      if(R<=R0v) then
       W1=3*P(1)+R*(-a*P(1)+4*P(2)+R*(-a*P(2)+5*P(3)+R*(-a*P(3)+6*P(4)+&
         R*(-a*P(4)+7*P(5)+R*(-a*P(5)+8*P(6)+R*(-a*P(6)+9*P(7)+&
         R*(-a*P(7)+10*P(8)+R*(-a*P(8)+11*P(9)+R*(-a*P(9)+12*P(10)-&
         a*R*P(10))))))))))
       W2=3*Q(1)+R*(-(b*Q(1))+4*Q(2)+R*(-2*c*Q(1)-b*Q(2)+5*Q(3)+&
         R*(-2*c*Q(2)-b*Q(3)+6*Q(4)+R*(-2*c*Q(3)-b*Q(4)+7*Q(5)+&
         R*(-2*c*Q(4)-b*Q(5)+8*Q(6)+R*(-2*c*Q(5)-b*Q(6)+9*Q(7)+&
         R*(-2*c*Q(6)-b*Q(7)+10*Q(8)+R*(-2*c*Q(7)-b*Q(8)+11*Q(9)+&
         R*(-2*c*Q(8)-b*Q(9)+12*Q(10)+R*(-2*c*Q(9)-b*Q(10)+13*Q(11)+&
         R*(-2*c*Q(10)-b*Q(11)+14*Q(12)+R*(-2*c*Q(11)-b*Q(12)+15*Q(13)+&
         R*(-2*c*Q(12)-b*Q(13)+16*Q(14)+R*(-2*c*Q(13)-b*Q(14)+17*Q(15)+&
         R*(-2*c*Q(14)-b*Q(15)+18*Q(16)+R*(-2*c*Q(15)-b*Q(16)+19*Q(17)+&
         R*(-2*c*Q(16)-b*Q(17)+20*Q(18)+R*(-2*c*Q(17)-b*Q(18)+21*Q(19)+&
         R*(-2*c*Q(18)-b*Q(19)+22*Q(20)+R*(-2*c*Q(19)-b*Q(20)+23*Q(21)+&
         R*(-2*c*Q(20)-b*Q(21)+24*Q(22)+R*(-2*c*Q(21)-b*Q(22)+25*Q(23)+&
         R*(-2*c*Q(22)-b*Q(23)+26*Q(24)+R*(-2*c*Q(23)-b*Q(24)+27*Q(25)+&
         R*(-2*c*Q(24)-b*Q(25)-2*c*R*Q(25))))))))))))))))))))))))))
       PV=EXP(-a*R)*W1+EXP(-(b+c*R)*R)*W2
      else
        iR=1.Q0/R
        iRs=iR*iR
        PV=iR*iRs*iRs*iRs*(Nv(7)+iR*(Nv(8)+iR*(Nv(9)+iR*(Nv(10)+&
          iR*(Nv(11)+iR*(Nv(12)+iR*(Nv(13)+iR*(Nv(14)+iR*(Nv(15)+&
          iR*(Nv(16)+iR*Nv(17)))))))))))
      endif
      return
      end function PV

!----------------------------------------------------------------------

      real*16 function RPdV(R)
!Evaluates R * ( 2/R + d/dR ) dV(R) * mu_n^2.
!See Eq. 34 in J. Komasa, M. Puchalski, P. Czachorowski, G. Lach, K. Pachucki, Phys. Rev. A 100, 032519 (2019)
!From K. Pachucki and J. Komasa, J. Chem. Phys. 143, 034111 (2015) (supplement)
      implicit none
      real*16 :: R,W1,W2,a,b,c,R0dv,iR,iRs
      real*16, dimension(:) :: P(0:7),Q(30),Ndv(7:17)
          R0dv=10.478Q0
          a=1.476341944091356128570472825054749Q0
          b=5.228765110685560964176093401049546Q0
          c=0.5489643581419930074834090988043598Q0
          P(0)=-2.612046016128400967420886395954259Q0
          P(1)=3.123320575839294037926093781973416Q1
          P(2)=-1.999049672593699107695428947612856Q1
          P(3)=5.643790806049411464753215415109306Q0
          P(4)=-8.718918597597365722585784215420987Q-1
          P(5)=7.710079330866428678492988175720208Q-2
          P(6)=-3.692609173333851500954795390310001Q-3
          P(7)=7.510836080860432964711403436419208Q-5
          Q(1)=-2.050781379319505444765981229948498Q1
          Q(2)=-6.245671638845641302554561532124580Q1
          Q(3)=-4.673492671334046348301460641020805Q2
          Q(4)=2.775290571132609056357502107874890Q3
          Q(5)=-1.929611381985240594373174137423261Q4
          Q(6)=9.486435593226647135886894233406023Q4
          Q(7)=-3.743283541467850201595778911678327Q5
          Q(8)=1.192110467673869169198529282979016Q6
          Q(9)=-3.117442537011975429900210219940664Q6
          Q(10)=6.765755566331404374767101448795065Q6
          Q(11)=-1.229273067258218857100837283326112Q7
          Q(12)=1.882178777092897955834911185577372Q7
          Q(13)=-2.440844475331365804283130356550500Q7
          Q(14)=2.690778129358984706352418985140552Q7
          Q(15)=-2.527792976959836632812732205008668Q7
          Q(16)=2.026317978111770857846935806661185Q7
          Q(17)=-1.386351609952432355012821151514896Q7
          Q(18)=8.088045741672726308556742880177882Q6
          Q(19)=-4.015064577478505698283630839568315Q6
          Q(20)=1.690035658952137274946405742927352Q6
          Q(21)=-6.000983869268033953974372291990401Q5
          Q(22)=1.784714274210052388475871565082275Q5
          Q(23)=-4.402551318121617125468514761250021Q4
          Q(24)=8.889420042304083810330753020786004Q3
          Q(25)=-1.442571663028314522021450798848065Q3
          Q(26)=1.833277807348720030744375749325572Q2
          Q(27)=-1.755306480182845490937931075564161Q1
          Q(28)=1.189626720930838570495404118347148Q0
          Q(29)=-5.083904033446884557571579460238142Q-2
          Q(30)=1.029858651727620803709825502407178Q-3

          Ndv(7)=-2.946715853536231413299547859031183Q0
          Ndv(8)=-4.735124517844690869537095092186823Q0
          Ndv(9)=-1.070917239073512330236213151105970Q1
          Ndv(10)=-2.861091089433546100552926773334360Q1
          Ndv(11)=-3.796613028303815163230470934784121Q1
          Ndv(12)=-4.313266654091046099132698746351802Q1
          Ndv(13)=-4.794930562796385162865826637084648Q1
          Ndv(14)=-5.274532584373873393769437025323082Q1
          Ndv(15)=-2.059203927162487259336871298791335Q9
          Ndv(16)=1.576760752900018996256365461658068Q10
          Ndv(17)=-3.221663640419939369495553795255815Q10
      if(R<=R0dv) then
        W1=2*P(0)+R*(-a*P(0)+3*P(1)+R*(-a*P(1)+4*P(2)+R*(-a*P(2)+5*P(3)+&
          R*(-a*P(3)+6*P(4)+R*(-a*P(4)+7*P(5)+R*(-a*P(5)+8*P(6)+&
          R*(-a*P(6)+9*P(7)-a*R*P(7))))))))
        W2=3*Q(1)+R*(-(b*Q(1))+4*Q(2)+R*(-2*c*Q(1)-b*Q(2)+5*Q(3)+&
       R*(-2*c*Q(2)-b*Q(3)+6*Q(4)+R*(-2*c*Q(3)-b*Q(4)+7*Q(5)+&
       R*(-2*c*Q(4)-b*Q(5)+8*Q(6)+R*(-2*c*Q(5)-b*Q(6)+9*Q(7)+&
       R*(-2*c*Q(6)-b*Q(7)+10*Q(8)+R*(-2*c*Q(7)-b*Q(8)+11*Q(9)+&
       R*(-2*c*Q(8)-b*Q(9)+12*Q(10)+R*(-2*c*Q(9)-b*Q(10)+13*Q(11)+&
       R*(-2*c*Q(10)-b*Q(11)+14*Q(12)+R*(-2*c*Q(11)-b*Q(12)+15*Q(13)+&
       R*(-2*c*Q(12)-b*Q(13)+16*Q(14)+R*(-2*c*Q(13)-b*Q(14)+17*Q(15)+&
       R*(-2*c*Q(14)-b*Q(15)+18*Q(16)+R*(-2*c*Q(15)-b*Q(16)+19*Q(17)+&
       R*(-2*c*Q(16)-b*Q(17)+20*Q(18)+R*(-2*c*Q(17)-b*Q(18)+21*Q(19)+&
       R*(-2*c*Q(18)-b*Q(19)+22*Q(20)+R*(-2*c*Q(19)-b*Q(20)+23*Q(21)+&
       R*(-2*c*Q(20)-b*Q(21)+24*Q(22)+R*(-2*c*Q(21)-b*Q(22)+25*Q(23)+&
       R*(-2*c*Q(22)-b*Q(23)+26*Q(24)+R*(-2*c*Q(23)-b*Q(24)+27*Q(25)+&
       R*(-2*c*Q(24)-b*Q(25)+28*Q(26)+R*(-2*c*Q(25)-b*Q(26)+29*Q(27)+&
       R*(-2*c*Q(26)-b*Q(27)+30*Q(28)+R*(-2*c*Q(27)-b*Q(28)+31*Q(29)+&
       R*(-2*c*Q(28)-b*Q(29)+32*Q(30)+R*(-2*c*Q(29)-b*Q(30)-2*c*R*Q(30)&
       ))))))))))))))))))))))))))))))
        RPdV=EXP(-a*R)*W1+EXP(-(b+c*R)*R)*W2*R
      else
        iR=1.Q0/R
        iRs=iR*iR
        RPdV=iRs*iRs*iRs*(Ndv(7)+iR*(Ndv(8)+iR*(Ndv(9)+iR*(Ndv(10)+&
          iR*(Ndv(11)+iR*(Ndv(12)+iR*(Ndv(13)+iR*(Ndv(14)+iR*(Ndv(15)+&
          iR*(Ndv(16)+iR*Ndv(17)))))))))))
      endif
      return
      end function RPdV

!----------------------------------------------------------------------

      real*16 function RU(R)
!Evaluates R * U(R) * mu_n^2.
!See Eq. 34 in J. Komasa, M. Puchalski, P. Czachorowski, G. Lach, K. Pachucki, Phys. Rev. A 100, 032519 (2019)
!From K. Pachucki and J. Komasa, J. Chem. Phys. 143, 034111 (2015) (supplement)
      implicit none
      real*16 :: R,W1,W2,a,b,c,R0u,iR,iRs
      real*16, dimension(:) :: P(0:16),Q(20),Au(6:16)
          R0u=10.943Q0
          a=1.533800221775688781423963317520503Q0
          b=4.358275759106648506920343788268704Q0
          c=0.8430742511875004421673637531811377Q0
          P(0)=-16.11474950423569226444606367034071Q0
          P(1)=-1.597801610512121516538297241006934Q6
          P(2)=3.309012953430007557625476187883853Q6
          P(3)=-3.186989368644250627770718958819695Q6
          P(4)=1.892777934852434839652472277637295Q6
          P(5)=-7.749059899799896754519771985968581Q5
          P(6)=2.315609965877670146726200877641217Q5
          P(7)=-5.215879404994756797445257052756161Q4
          P(8)=9.015263509093277075687292841973858Q3
          P(9)=-1.205227644354326886525074626539829Q3
          P(10)=1.245967969466150826562658384007904Q2
          P(11)=-9.877655544992889918315509694081757Q0
          P(12)=5.896243426613885281163906552063623Q-1
          P(13)=-2.565037087203693575718160691933129Q-2
          P(14)=7.676512564757049232588616283788814Q-4
          P(15)=-1.413119042316880145424605800258526Q-5
          P(16)=1.206138706042188809267317191945814Q-7
          Q(1)=1.597886117345573340487559855836458Q6
          Q(2)=1.204063360571316145382772926775498Q6
          Q(3)=1.559408593864920082212296511612249Q6
          Q(4)=9.392169665840891928994615239509150Q5
          Q(5)=6.333320911242088238148796847173335Q5
          Q(6)=5.782063193337288858669644698516552Q5
          Q(7)=-4.207052275513858945306180567653897Q5
          Q(8)=1.373472841002073724121895488788441Q6
          Q(9)=-2.097108778311422269267227613178391Q6
          Q(10)=2.849102195158770217263932197958446Q6
          Q(11)=-3.014545597394349568799170214719517Q6
          Q(12)=2.590490510938653350371546092256476Q6
          Q(13)=-1.777208095412428484541316917535973Q6
          Q(14)=9.722842708641441123371307057449953Q5
          Q(15)=-4.177041267621820444207912981135484Q5
          Q(16)=1.383787560571193898326691726404061Q5
          Q(17)=-3.417703171462872442682541553547207Q4
          Q(18)=5.966348409998999196726430114135514Q3
          Q(19)=-6.611919925753985709910374848840511Q2
          Q(20)=3.589295330696601412111845975144198Q1
          
          Au(6)=1.764271521315327216535667311817119Q1
          Au(7)=2.732210499170234560680035149724180Q1
          Au(8)=2.381553452943955964430713997354216Q2
          Au(9)=8.550513524988028753383410779231207Q2
          Au(10)=3.723555774552514856004914957924202Q3
          Au(11)=1.115987288837899988434482477118217Q4
          Au(12)=1.361082970183686725988365038417660Q4
          Au(13)=1.375959814875091325759964558591339Q4
          Au(14)=-8.613110080358075747651036003325528Q9
          Au(15)=2.550607210203041871442573986328024Q11
          Au(16)=-1.332077797360706174112478754799057Q12
      if(R<=R0u) then
        W1=P(0)+R*(P(1)+R*(P(2)+R*(P(3)+R*(P(4)+R*(P(5)+R*(P(6)+&
          R*(P(7)+R*(P(8)+R*(P(9)+R*(P(10)+R*(P(11)+R*(P(12)+R*(P(13)+&
          R*(P(14)+R*(P(15)+R*P(16))))))))))))))))
        W2=R*(Q(1)+R*(Q(2)+R*(Q(3)+R*(Q(4)+R*(Q(5)+R*(Q(6)+R*(Q(7)+&
          R*(Q(8)+R*(Q(9)+R*(Q(10)+R*(Q(11)+R*(Q(12)+R*(Q(13)+R*(Q(14)+&
          R*(Q(15)+R*(Q(16)+R*(Q(17)+R*(Q(18)+R*(Q(19)+R*Q(20)&
           )))))))))))))))))))
        RU=EXP(-a*R)*W1+EXP(-(b+c*R)*R)*W2
        RU=RU*R
      else
        iR=1.Q0/R
        iRs=iR*iR
        RU=-iR*iRs*iRs*(Au(6)+iR*(Au(7)+iR*(Au(8)+iR*(Au(9)+iR*(Au(10)+&
          iR*(Au(11)+iR*(Au(12)+iR*(Au(13)+iR*(Au(14)+iR*(Au(15)+&
          iR*Au(16)))))))))))
      endif
      return
      end function RU

!----------------------------------------------------------------------
      real*16 function VEnaprim(R)
!Evaluates heteronuclear contribution to the potential, dEnaprim(R)/lambda^2.
!Analytic fit by G.Lach, based on new calculations,
!J. Komasa, M. Puchalski, P. Czachorowski, G. Lach, K. Pachucki, Phys. Rev. A 100, 032519 (2019).
!Confer Eq. 37 in the above work. 
      implicit none
      real*16, intent(in)::R
      real*16 :: F
      real*16 :: A0=-0.6915960872275q0
      real*16 :: A1 = 2.29052040999132901736873207q0
      real*16 :: A2 = -4.89578473682906565214202544q0
      real*16 :: A3 = 3.16615086740974133635197863q0
      real*16 :: A4 = -0.774172046034086263480052565q0
      real*16 :: c = -3.45691286596611291673537828q0
      real*16 :: bb = 3.31720281745853229268558648q0
      real*16 :: B6 = 133.7168923197120405200422694q0
      real*16 :: b = -1.318510060789572974067739745q0
 
      VEnaprim=(A0+A1*R+A2*R**2+A3*R**3+A4*R**4)/R**4*Exp(-c*R+b*R**2)+F(6,bb*R)*B6/R**6
      end

!----------------------------------------------------------------------
      subroutine interp(init_file,NV,sep,ord,asymp,pow,potV)
      !Potential interpolation by PPPACK (http://www.netlib.org/pppack)
      !(based on 'A Practical Guide to Splines' by C. de Boor; see particular files in './PPPACK' or details)
      implicit none
      integer,parameter :: maxpoints=1000 !Max number of initial points to interpolate
      CHARACTER( LEN = 30 ),intent(in) :: init_file !Name of file with data
      integer, intent(in) :: ord !Interpolation order 
      integer, intent(in) :: NV !Number of DVR points where the resulting function will be evaluated
      integer, intent(in) :: pow !The data values will be multiplied with x(i)**pow before interpolation
      double precision,intent(in)::sep !Separation of the grid where the function will be evaluated
      double precision, intent(in):: asymp !x-> infinity asymptotic value to subtract
      double precision,intent(out) :: potV(NV) !DVR representation of the interpolated potential
      double precision :: bvalue !PPPACK function which evaluates the interpolated potential, './PPPACK/bvalue.f'
      double precision, allocatable, dimension(:) :: tau,scrtch,bcoef,q,t
      double precision, allocatable, dimension(:) :: potread
      integer :: lngth, stat,i 
      real(16):: x(maxpoints),V(maxpoints)
      logical :: fileexists


      !Check if the data file exists at all
      inquire(file=trim(init_file),exist=fileexists)
      if (fileexists.eqv..false.) then
       write(*,*)"ERROR! File ",trim(init_file)," does not exist!"
       stop
      endif

      !Open it and read
      OPEN(UNIT=9,FILE=init_file,FORM='FORMATTED')
      do i=1,maxpoints
       READ(9,*,iostat=stat) x(i),V(i)
       if (stat == 0) then
       lngth=i
       else
        exit
       endif
      enddo
      close(9)

      !Prepare arrays
      allocate(tau(lngth),scrtch((lngth-ord)*(2*ord+3)+5*ord+3), bcoef(lngth))
      allocate(q((2*ord-1)*lngth),t(lngth+ord),potread(lngth))

      !Subtract the asymptotic value and multiply with x(i)**pow (can improve stability, e.g. for E(4,0)(R))
      do i=1,lngth
      tau(i)=x(i)
      potread(i)=(V(i)-asymp)*x(i)**pow
      enddo
      !Call PPPACK to interpolate
      call splopt(tau,lngth,ord,scrtch,t,stat)
      call splint(tau,potread,t,lngth,ord,q,bcoef,stat)
      !Return the value of the potential on the DVR grid points
      do i=1, NV
       potV(i)=bvalue(t,bcoef,lngth,ord,sep*i,0)/(sep*i)**pow

      enddo

      end

!-----------------------------------------------------------------------------------------------
      subroutine loader(p,lambda,alpha,dr,N,na_cutoff)
!Reads the values of correction potentials on a DVR grid
!or calculates them from analytic fits.
!If not already taken into account in the source file, 
!the asymptotic values for an infinite distance are subtracted.
      use h2spectr_types
			use control_parameters
      implicit none
      character(len=30) :: infile
      double precision, intent(in)::lambda,alpha,dr,na_cutoff
      integer,intent(in)::N
      type(potentials),intent(out)::p
      integer :: i
      real*16 :: x,V,Eaint,WTint,VEna,VEnaprim,f
      double precision :: delta12(N),araki(N),logK(N),pi,lnKH
      parameter (pi=3.14159265358979324d0)
      parameter (lnKH=2.984128555765498q0)
      !lnKH is Bethe logarithm for H, from G. W. F. Drake, and R. A. Swainson, Phys. Rev. A 41, 1243(1990)
      real*16 :: R,Rinv,Rinvq,Rinv6,Rinv12
      
      call allocpot(p,N)

      infile="data/ma4.dat" !BO relativistic E(4,0)(R) potential (Eq. 60)
      call interp(infile,N,dr,6,-0.25d0,4,p%ma4)

      infile="data/ma4na.dat" !Relativistic nonadiabatic 2*mu_n*E(4,1)(R) potential (Eq. 62)
      call interp(infile,N,dr,3,0.d0,0,p%ma4na)

      infile="data/ma6.dat" !HQED E(6,0)(R) potential (Eq. 73)
      call interp(infile,N,dr,6,2*(-1.d0/16 + 3.0616225d0 + 0.054606d0),0,p%ma6)

      infile="data/delta_ia.dat" !Electron-nucleus Dirac deltas for E(5,0)(R), E(7)(R) and E(4)_FS(R) (Eqs. 70, 76 and 78)
      call interp(infile,N,dr,6,2/pi,0,p%deltaia)

      !Leading QED E(5,0)(R) components (confer Eq. 70)
      infile="data/lnK.dat" !Bethe logarithm
      call interp(infile,N,dr,6,0.d0,0,logK)
      infile="data/araki.dat" !Araki-Sucher term
      call interp(infile,N,dr,6,0.d0,0,araki)
      infile="data/delta_12.dat" !Electron-electron Dirac delta
      call interp(infile,N,dr,6,0.d0,0,delta12)

      ! Hyperfine parameters
			if(HF_bool) then
				if(molecule=='HD') then
          infile="data/cpHD.dat" ! cp(R) hyperfine potential of HD
          call interp(infile,N,dr,4,0.d0,3,p%HFcp)
          infile="data/cdHD.dat" ! cd(R) hyperfine potential of HD
          call interp(infile,N,dr,4,0.d0,3,p%HFcd)
          !infile="data/qdHD.dat" ! q(R) electric field gradient potential of HD
          !call interp(infile,N,dr,4,0.d0,3,p%HFqd)
          infile="data/d2HD.dat" ! d2(R) hyperfine potential of HD
          call interp(infile,N,dr,4,0.d0,3,p%HFd2)
        endif
			 	if(molecule=='HT') then
          infile="data/cpHT.dat" ! cp(R) hyperfine potential of HT
          call interp(infile,N,dr,4,0.d0,3,p%HFcp)
          infile="data/ctHT.dat" ! ct(R) hyperfine potential of HT
          call interp(infile,N,dr,4,0.d0,3,p%HFcd) ! cd is used to store also HT data
      	endif
			 	if(molecule=='H2') then
          infile="data/cpH2.dat" ! cp(R) hyperfine potential of H2
          call interp(infile,N,dr,4,0.d0,3,p%HFcp)
      	endif
			 	if(molecule=='D2') then
          infile="data/cdD2.dat" ! cd(R) hyperfine potential of D2
          call interp(infile,N,dr,4,0.d0,3,p%HFcd)
      	endif
			endif

      do i=1,N
      !Use analytic fits where available
      x=dr*i
      p%R(i)=x !Grid points
      p%V(i)=V(x) !BO nonrelativistic potential E_el(R) (Eq. 12)
      p%Vad(i)=Eaint(x) !Adiabatic nonrelativistic contribution 2*mu_n*E_a(R) (Eq. 18)
      p%Worto(i)=WTint(x) !W(R)*mu_n^2 perpendicular (Eq. 29)
      p%VEna(i)=VEna(x) !Nonadiabatic correction to potential (Eq. 34)
      p%VEnaprim(i)=VEnaprim(x)*lambda**2 !Heteronuclear contribution to potential (Eq. 37)
      call WIIint(x,p%Wpara(I),p%dWpara(I),p%d2Wpara(I)) !W(R)*mu_n^2 parallel and its first two derivatives over R (Eq. 28)
      !Combine the leading QED E(5) components (Eq. 70) and subtract the asymptotic (2H value)
       p%ma5(i)=4.d0/3*(19.d0/30-2*Log(alpha)-logK(i))*(p%deltaia(i)+2/pi)&
               +(164.d0/15+14.d0/3*Log(alpha))*delta12(i)-7.d0/(6*pi)*araki(i)
       p%ma5(i)=p%ma5(i)-2*(4.d0/3*(19.d0/30-2*log(alpha)-lnKH)/pi)

      !Short range cutoffs
      if (x.lt.na_cutoff) then
        p%VEnaprim(i)=VEnaprim(1.q0*na_cutoff)*lambda**2
      endif
      !Long-distance asymptotic fits (switching points chosen for x such that the value of the fit and the interpolating polynomial,
      !as well as their first derivatives, don't differ much).
      if (x.gt.6.3q0) p%ma4(i)=18612.50377457077d0/x**10 + 2755.838506596614d0/x**8 - 12.666930848516696d0/x**6 + 0.462806538843d0/x**4
      if (x.gt.5.6q0) p%ma4na(i)=2802.163028d0/x**8 + 40.74499126d0/x**6 + 3.025784956d0/x**4
      if (x.gt.7.1q0) p%ma5(i)=-73488.61425836063d0/x**8 - 27.852115041081685d0/x**7 + 480.1507456397395d0/x**6 - 2.228169203286535d0/x**5&
                               - 0.37136153388108917d0/x**3
      if (x.gt.5.85q0) p%ma6(i)=-3097.0974663204775d0/x**6 + 53.135323927439366d0/x**5 + 35.974297438942074d0/x**4 + 0.529947904d0/x**2
      if (x.gt.7.q0) p%deltaia(i)=-253384.50421647355d0/x**10 + 1290.4050762253792d0/x**8 - 22.08150337973702d0/x**6

        ! Hyperfine parameters
  			if(HF_bool) then
          R=x
          Rinv=1.d0/R
          Rinvq=Rinv*Rinv*Rinv
          Rinv6=Rinvq*Rinvq
          Rinv12=Rinv6*Rinv6
					if(molecule=='HD') then
            p%HFd1(i)=49.77349654141856d0/x**3
            if(x>5.011d0) p%HFcp(i)=(-8.40042824451842634796136082677d7+R*(4.00957663318886517928455765495d7+R*(-5.56111092915328413112089165464d6+R*241345.1575956017409168464166176d0)))*Rinv12*R
            if(x>5.012d0) p%HFcd(i)=(-1.521785912240564997708188245809d7+R*(7.14361783474054255859580208210d6+R*(-966611.408602756815487686985639d0+R*40917.2609809419819920512202549d0)))*Rinv12*R
            !if(x>5.013d0) p%HFqd(i)=(614955.151162155691828770084403d0+R*(-231765.1896111859488705153924952d0+R*(21634.62892597528592073269454520d0-R*714.276078271861013542195327522d0)))*Rinv12*R
            if(x>5.013d0) p%HFd2(i)=(-4.12667209342213814067655749967d7+R*(1.555266165976024845151026739085d7+R*(-1.451797245240479223950607402798d6+R*47931.6768651034078808253455792d0)))*Rinv12*R
        	endif
					if(molecule=='HT') then
            p%HFd1(i)=345.8520344001768d0/x**3
            if(x>5.011d0) p%HFcp(i)=(-7.7357297788066787833149884711d7+R*(3.6785736580272326185323918529d7+R*(-5.0743349338279123783079017026d6+R*219037.30911778361079496728212d0)))*Rinv12*R
            if(x>5.005d0) p%HFcd(i)=(-5.9582246183614896748688596625d7+R*(2.9476721583857585482267966803d7+R*(-4.2974493054266213214225013862d6+R*195435.23106535271073253605084d0)))*Rinv12*R
        	endif
					if(molecule=='H2') then
            p%HFd1(i)=324.2444136993476d0/x**3
            if(x>5.010d0) p%HFcp(i)=(-1.0398945460909662775852895192d8+R*(5.0047875124992303755527115890d7+R*(-7.0246771285156198467710204988d6+R*308417.10306162114922212668160d0)))*Rinv12*R
        	endif
					if(molecule=='D2') then
            p%HFd1(i)=7.640535513607182d0/x**3
            if(x>5.013d0) p%HFcd(i)=(-1.2150013214957769842813200822d7+R*(5.6159083919101899846611357432d6+R*(-741945.0640665495430753244075d0+R*30621.307975639871298781219565d0)))*Rinv12*R
        	endif
        endif
      enddo
      end

