(* SmeftFR v3.0 package *)
(* I/O routines: code for generating FeynRules parameter files with
internal and external parameters *)



SMEFTParFile = Function[{oplist, TargetFile},
(* generates FeynRules par file including only "active" operators and internal parameters *)
Module[{tmp,date,cfile},

Print["Parameter file in Warsaw basis generated as ", Style[ TargetFile, Bold ] ];

cfile = OpenWrite[TargetFile];

WriteLine[cfile, "(* Generated by SMEFTParFile routine *)"];
date = Date[];
WriteLine[cfile, "(* " <> ToString[date[[3]]] <> "." <>
    ToString[date[[2]]] <> "." <> ToString[date[[1]]] <> " " <>
    ToString[date[[4]]] <> ":" <> ToString[date[[5]]] <> " *)\n"]

WriteLine[cfile, "(* Active operators included in Feynman Rules: " <> ToString[oplist] <> " *)\n"];

(* copy header with fixed definitions first *)
WriteString[cfile, ReadString[FileNameJoin[{SMEFT$Path, "definitions", "smeft_par_head_WB.fr"}]] ];
WriteLine[cfile, "\n"];

(* add input observables in preferred scheme to fix SM Lagrangian parameters;
see file smeft_input_parameters.m *)
WriteLine[cfile, "(* input observables *)\n"];
WriteInputParameter[cfile,#] & /@ SM$InputParameters;

WriteLine[cfile, "(* dim 6 couplings in Warsaw basis *)\n"];
WriteLine[cfile, "(* flavor independent *)\n"];
If[ MemberQ[oplist, # ], WriteInternalScalarEntry[ cfile, # ] ] & /@ ScalarWC;

WriteLine[cfile, "(* flavor dependent *)\n"];
WriteLine[cfile, "(* 2 fermion operators *)\n"];
If[ MemberQ[oplist, #[[1]] ] && ! #[[5]], WriteInternalTensor2Entry[ cfile, # ] ] & /@ Tensor2WC;

WriteLine[cfile, "(* 4 fermion operators *)\n"];
If[ MemberQ[oplist, #[[1]] ] && ! #[[6]], WriteInternalTensor4Entry[ cfile, # ] ] & /@ Tensor4WC;

WriteLine[cfile, "(* L or B violating operators *)\n"];
If[ MemberQ[oplist, #[[1]] ] && #[[5]], WriteInternalTensor2Entry[ cfile, # ] ] & /@ Tensor2WC;
If[ MemberQ[oplist, #[[1]] ] && #[[6]], WriteInternalTensor4Entry[ cfile, # ] ] & /@ Tensor4WC;

WriteLine[cfile, "(* redefined (mass basis) WC coefficients *)\n"];
WriteLine[cfile, "(* flavor independent *)\n"];
If[ MemberQ[oplist, # ], WriteInternalMassScalarEntry[ cfile, # ] ] & /@  ScalarWC;

WriteLine[cfile, "(* flavor dependent *)\n"];
WriteLine[cfile, "(* 2 fermion operators *)\n"];
If[ MemberQ[oplist, #[[1]] ] && ! #[[5]], WriteInternalMassTensor2Entry[ cfile, # ] ] & /@ Tensor2WC;

WriteLine[cfile, "(* 4 fermion operators *)\n"];
If[ MemberQ[oplist, #[[1]] ] && ! #[[6]], WriteInternalMassTensor4Entry[ cfile, # ] ] & /@ Tensor4WC;

WriteLine[cfile, "(* L or B violating operators *)\n"];
If[ MemberQ[oplist, #[[1]] ] && #[[5]], WriteInternalMassTensor2Entry[ cfile, # ] ] & /@ Tensor2WC;
If[ MemberQ[oplist, #[[1]] ] && #[[6]], WriteInternalMassTensor4Entry[ cfile, # ] ] & /@ Tensor4WC;

WriteLine[cfile, "(* NP scale *)\n"];
WriteLine[cfile, "Lam == {"];
WriteLine[cfile, "  ParameterType -> Internal,"];
WriteLine[cfile, "  TeX           -> 1/\[CapitalLambda]^2,"];
WriteLine[cfile, "  Description   -> \"Effective NP scale squared\""];
WriteLine[cfile, "}\n"];

WriteLine[cfile, "}\n"];

(* updates the correct maximal NP interaction order for model files *)
Get[ FileNameJoin[{SMEFT$Path, "definitions", "smeft_par_io.fr"}] ];
WriteLine[cfile, "M$InteractionOrderHierarchy = ", M$InteractionOrderHierarchy, "\n"];
WriteLine[cfile, "M$InteractionOrderLimit = ", M$InteractionOrderLimit /. {NP,tmp_} -> {NP,SMEFT$ExpansionOrder}, "\n"];
       
Close[TargetFile];

]
(* end of SMEFTParFile *)
]





(* export routine, store Lagrangian and Feynman in mass basis to file *)
SMEFTOutput[ OptionsPattern[ {TargetFile -> FileNameJoin[{ SMEFT$Path, "output", "smeft_feynman_rules.m" }],
			      ModelFile -> FileNameJoin[{ SMEFT$Path, "output", "smeft_par_MB.fr" }],
			      OverwriteTarget -> True,
			      Operators -> SMEFT$OperatorList } ] ] :=
Module[{cfile,vlist,laglist,clist,ylist,slist,date,il,tmp,i,j},

vlist = {"LeptonGauge", "LeptonHiggsGauge", "QuarkGauge",
"QuarkHiggsGauge", "QuarkGluon", "GaugeSelf", "GaugeHiggs",
"GluonSelf", "GluonHiggs", "FourLepton", "TwoQuarkTwoLepton",
"FourQuark", "BLViolating"};

laglist = Join[ # <> "Lagrangian" & /@ vlist, {"DeltaLTwoLagrangian"} ];
vlist   = # <> "Vertices" & /@ vlist;

ylist = "SMEFT$" <> # & /@ {"YL", "YD", "YU"};

slist = "SMEFT$" <> # & /@ {"OperatorList", "RxiGaugeStatus",
"MajoranaNeutrino", "Correct4FermionSign", "WB", "MB",
"ExpansionOrder"};

(* generate parameter files in mass basis *)
WCXFToSMEFT[ OptionValue[ModelFile],
        Operators -> OptionValue[Operators],
	OverwriteTarget -> True,
	RealParameters -> SMEFT$RealParameters ];

(* check options *)
If [ FileExistsQ[ OptionValue[TargetFile] ] && (! OptionValue[OverwriteTarget]),
    Print["File " <> OptionValue[TargetFile] <> " exists!"];
    TargetExistsQ = AskFunction[ Ask["Overwrite (y/n)?"-> "String"] ][];
];
If[ TargetExistsQ != "y", Print[ "Change name of output file and rerun" ]; Abort[]; ];
Print["Output file will be generated as ",  Style[OptionValue[TargetFile], Bold ] ];

cfile = OpenWrite[OptionValue[TargetFile]];

WriteLine[cfile, "(* Generated by SMEFTOutput routine *)"];
date = Date[];
WriteLine[cfile, "(* " <> ToString[date[[3]]] <> "." <>
    ToString[date[[2]]] <> "." <> ToString[date[[1]]] <> " " <>
    ToString[date[[4]]] <> ":" <> ToString[date[[5]]] <> " *)\n"]
WriteLine[cfile, "(* Active operators included in Lagrangian and Feynman rules: " <> ToString[SMEFT$OperatorList] <> " *)\n"];

WriteLine[cfile, "(* interaction vertices in mass basis *)\n"];
WriteLineEntry[cfile,  # <> " = ", ToExpression[ # ], "\n\n" ] & /@ vlist;
If[ SMEFT$RxiGaugeStatus, WriteLineEntry[cfile, "GhostVertices = ", GhostVertices, "\n\n" ] ];

WriteLine[cfile, "(* Lagrangian in mass basis *)\n"];
WriteLineEntry[cfile,  # <> " = ", ToExpression[ # ], "\n\n" ] & /@ laglist;
If[ SMEFT$RxiGaugeStatus, WriteLineEntry[cfile, "GhostLagrangian = ", GhostLagrangian, "\n\n" ] ];

WriteLine[cfile, "(* corrections to fields and couplings *)\n"];
WriteLine[cfile, "(* SMEFT scheme *)\n"];
NormalizationConstantsExpansion["smeft"];
WriteLineEntry[cfile, "Default$InputSub =", SMEFT$InputSub, "\n\n" ]
WriteLine[cfile, "(* User scheme *)\n"];
NormalizationConstantsExpansion["user"];
WriteLineEntry[cfile, "User$InputSub = ", SMEFT$InputSub, "\n\n" ]
WriteLineEntry[cfile, "SMEFT$InputListStatus = ", True, "\n\n" ]

(* Yukawa couplings *)
Clear[i,j];
WriteLineEntry[cfile,  # <> "[i,j] = ", OptimizeIndex[ ToExpression[ #  <> "[i,j]" ] ], "\n\n" ] & /@ ylist;

WriteLine[cfile, "(* setup variables *)\n"];
WriteLineEntry[cfile,  # <> " = ", ToExpression[ # ], "\n\n" ] & /@ slist;

WriteLineEntry[cfile,  "FeynmanGauge = ",  SMEFT$RxiGaugeStatus, "\n\n" ];

Close[OptionValue[TargetFile]];

]
(* end of SMEFTOutput *)




SMEFTLoadModelFiles = Function[{SMEFTParName},
(* load to FeynRules SMEFT field and parameter definitions *)
Get[ FileNameJoin[{SMEFT$Path, "definitions", "smeft_fields_WB.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "output", SMEFTParName}] ];

LoadModel[];
			
(* end of SMEFTLoadModelFiles *)
]




SMEFTLoadLagrangian = Function[{},
(* loads SM and dim6 operator classes *)

(* set gauge *)
SMEFTGaugeRules = If[ ! SMEFT$RxiGaugeStatus, {G0|GP|GPbar ->0}, {} ];
(* Load SM Lagrangian *)
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "00_sm.fr"}] ];
(* Load dim6 operators *)
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "01_X3.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "02_phi6.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "03_psi2phi3.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "04_X2phi2.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "05_psi2Xphi.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "06_psi2phi2D.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "07_LlLl.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "08_RrRr.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "09_LlRr.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "10_LrRl.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "11_BL.fr"}] ];
(* Load dim8 operators *)
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "21_X4.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "22_phi8.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "23_phi6D2.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "24_phi4D4.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "25_X3phi2.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "26_X2phi4.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "27_X2phi2D2.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "28_Xphi4D2.fr"}] ];	

(* end of SMEFTLoadLagrangian *)
]




SMEFTToUFO[ Lagrangian_,
	    OptionsPattern[ {Output -> FileNameJoin[{SMEFT$Path, "output", "UFO"}],
			     CorrectIO -> True,
			     AddDecays -> False } ] ] :=
(* generates UFO output and corrects Interaction Order assignments for higher order vertices *)
Module[{target,input,output,outfile,line,char,x,y},

target = OptionValue[Output];
WriteUFO[ Lagrangian, Output -> target, AddDecays -> OptionValue[AddDecays] ];

(* remove from NP couplinds all unnecessary IO assignments *)
If [ OptionValue[CorrectIO] == True,

  Print[ Style[ "\nCorrecting non positive interaction order issues for NP vertices...", Red, Bold ] ];
  input   = FileNameJoin[{target, "couplings.py"}];
  output  = FileNameJoin[{target, "couplings.tmp"}];
  outfile = OpenWrite[output];

  char = 1;
  While[ char =!= EndOfFile,
    line = "";
    char = 1;
    While[ char != 10 && char =!= EndOfFile, (* 10 is char code of "\n" *)
      char  = BinaryRead[input];
      If [ char != 10 && char =!= EndOfFile, line = line <> FromCharacterCode[char] ] ;
    ];

    If [ StringContainsQ[line, "'NP':"],
      line = StringReplace[ line, "'NP':" ~~ x_ ~~ "," ~~ Shortest[y__] ~~ "})" ->
                                  "'NP':" ~~ x ~~ "})" ]
       ];

    If [ char =!= EndOfFile,  WriteLine[outfile,line] ];
  ];

  Close[input];
  Close[outfile];

  DeleteFile[ input ];
  CopyFile  [ output, input ];
  DeleteFile[ output ];

];

Print["\nVertices stored in UFO format in directory", Style[ OptionValue[Output] ,Bold] ];

(* end of SMEFTToUFO *)
]




(* *************************************************** *)


(* low-level functions writing model file entries for scalar and
tensor WC's *)

WriteLineEntry = Function[{cfile,p1,p2,p3},
(* single line output *)

  WriteString[cfile, p1];
  Write[cfile, p2 ];
  WriteString[cfile, p3];

(* end of WriteLineEntry *)
]



(*
WriteInputParameter = Function[{cfile,par},
(* parameter file entry for physical input observable *)

WriteLine[cfile, ToString[ par[[1]] ] <> " == {"];
WriteLineEntry[cfile, "  ", #, ","] & /@ par[[2]];
WriteLine[cfile, "  ComplexParameter -> False"];
WriteLine[cfile, "},\n\n"];

(* end of WriteInputParameter *)
]
*)



WriteInputParameter = Function[{cfile,par},
(* parameter file entry for physical input observable *)

WriteLine[cfile, ToString[ par[[1]] ] <> " == {"];
WriteLineEntry[cfile, "  ", #, ","] & /@ Drop[ par[[2]], -1 ];
WriteLineEntry[cfile, "  ", par[[ 2, Length[ par[[2]] ] ]], "},\n\n"];

(* end of WriteInputParameter *)
]



WriteInternalScalarEntry = Function[{cfile,wc},
(* parameter file entry for scalar Warsaw basis WC *)
(* wc argument is member of ScalarWC list *)
Module[{tex,name,pos},

name = ToString[wc];
pos = StringPosition[name, "phi",1];
tex = name;
If [Length[StringPosition[tex, "phi"]] != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];

WriteLine[cfile, SMEFT$WB <> name <> " == {"];
WriteLine[cfile, "  ParameterType    -> Internal,"];
WriteLine[cfile, "  Indices          -> {},"];
WriteLine[cfile, "  InteractionOrder -> {NP," <> ToString[ SMEFT$NPOrder[name] ] <> "},"];
WriteLine[cfile, "  Definitions      -> {" <> SMEFT$WB <> name <> " :> " <> SMEFT$MB <> name <> "},"];
WriteLine[cfile, "  ComplexParameter -> False,"];
WriteLine[cfile, "  TeX              -> Superscript[" <> SMEFT$WB <> ", " <> tex <> "],"];
WriteLine[cfile, "  Description      -> \"Q" <> name <> " WC coefficient\""];
WriteLine[cfile, "},\n\n"];

]
(* end of WriteInternalScalarEntry *)
]



WriteInternalTensor2Entry = Function[{cfile,wc},
(* par file entry for 2-fermion Warsaw basis WC *)
(* wc argument is member of Tensor2WC list *)
Module[{tex,name,pos},

name = ToString[wc[[1]]];
tex = name;
pos = StringPosition[tex, "phi",1];
If [Length[StringPosition[tex, "phi"]] != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];
pos = StringPosition[tex, "f",1];
If [Length[StringPosition[tex, "f"]]   != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];

WriteLine[cfile, SMEFT$WB <> name <> " == {"];
WriteLine[cfile, "  ParameterType    -> Internal,"];
WriteLine[cfile, "  Indices          -> {Index[Generation], Index[Generation]},"];
WriteLine[cfile, "  InteractionOrder -> {NP," <> ToString[ SMEFT$NPOrder[name] ] <> "},"];
WriteLine[cfile, "  Definitions      -> {" <> SMEFT$WB <> name <>
                 "[ff1_,ff2_] :> Module[{ind1,ind2}, " <>
                 If[ wc[[5]],"Conjugate[",""] <> ToString[wc[[2]]] <>
                 "[ff1,ind1]" <> If[wc[[5]],"] "," "] <> SMEFT$MB <> name <>
                 "[ind1,ind2] Conjugate[" <> ToString[wc[[3]]] <> "[ff2,ind2]] ]},"];
WriteLine[cfile, "  Hermitian        -> " <> ToString[ wc[[4]] ] <> ","];
WriteLine[cfile, "  TeX              -> Superscript[" <> SMEFT$WB <> ", " <> tex <> "],"];
WriteLine[cfile, "  Description      -> \"Q" <> name <> " WC coefficient\""];
WriteLine[cfile, "},\n\n"];

]
(* end of WriteInternalTensor2Entry *)			
]



WriteInternalTensor4Entry = Function[{cfile,wc},
(* par file entry for 4-fermion Warsaw basis WC *)
(* wc argument is member of Tensor4WC list *)
Module[{name},

name = ToString[wc[[1]]];

WriteLine[cfile, SMEFT$WB <> name <> " == {"];
WriteLine[cfile, "  ParameterType    -> Internal,"];
WriteLine[cfile, "  Indices          -> {Index[Generation], Index[Generation], Index[Generation], Index[Generation]},"];
WriteLine[cfile, "  InteractionOrder -> {NP," <> ToString[ SMEFT$NPOrder[name] ] <> "},"];
WriteLine[cfile, "  Definitions      -> {" <> SMEFT$WB <> name <>
                 "[ff1_,ff2_,ff3_,ff4_] :> Module[{ind1,ind2,ind3,ind4}, " <>
                 If[wc[[6]],"Conjugate[",""] <> ToString[wc[[2]]] <>
                 "[ff1,ind1]" <> If[wc[[6]],"] "," "] <>
                 If[wc[[6]],"Conjugate[",""] <> ToString[wc[[3]]] <>
                 "[ff3,ind3]" <> If[wc[[6]],"] "," "] <> SMEFT$MB <> name <>
                 "[ind1,ind2,ind3,ind4] Conjugate[" <> ToString[wc[[4]]] <> "[ff2,ind2]] Conjugate[" <>
                 ToString[wc[[5]]] <> "[ff4,ind4]] ]},"];
WriteLine[cfile, "  TeX              -> Superscript[" <> SMEFT$WB <> ", " <> name <> "],"];
WriteLine[cfile, "  Description      -> \"Q" <> name <> " WC coefficient\""];
WriteLine[cfile, "},\n\n"];

]
(* end of WriteInternalTensor4Entry *)			
]



WriteInternalMassScalarEntry = Function[{cfile,wc},
(* par file entry for scalar mass basis internal WC *)
(* wc argument is member of ScalarWC list *)
Module[{tex,name,pos},

name = ToString[wc];
pos = StringPosition[name, "phi",1];
tex = name;
If [Length[StringPosition[tex, "phi"]] != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];

WriteLine[cfile, SMEFT$MB <> name <> " == {"];
WriteLine[cfile, "  ParameterType    -> Internal,"];
WriteLine[cfile, "  Indices          -> {},"];
WriteLine[cfile, "  InteractionOrder -> {NP," <> ToString[ SMEFT$NPOrder[name] ] <> "},"];
WriteLine[cfile, "  ComplexParameter -> False,"];
WriteLine[cfile, "  TeX              -> Superscript[" <> SMEFT$MB <> ", " <> tex <> "],"];
WriteLine[cfile, "  Description      -> \"Q" <> name <> " WC coefficient\""];
WriteLine[cfile, "},\n\n"];

]
(* end of WriteInternalMassScalarEntry *)
]



WriteInternalMassTensor2Entry = Function[{cfile,wc},
(* par file entry for 2-fermion mass basis internal WC *)
(* wc argument is member of Tensor2WC list *)
Module[{tex,name,pos},

name = ToString[wc[[1]]];
tex = name;
pos = StringPosition[tex, "phi",1];
If [Length[StringPosition[tex, "phi"]] != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];
pos = StringPosition[tex, "f",1];
If [Length[StringPosition[tex, "f"]]   != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];

WriteLine[cfile, SMEFT$MB <> name <> " == {"];
WriteLine[cfile, "  ParameterType    -> Internal,"];
WriteLine[cfile, "  Indices          -> {Index[Generation], Index[Generation]},"];
WriteLine[cfile, "  InteractionOrder -> {NP," <> ToString[ SMEFT$NPOrder[name] ] <> "},"];
WriteLine[cfile, "  Hermitian        -> " <> ToString[ wc[[4]] ] <> ","];
WriteLine[cfile, "  TeX              -> Superscript[" <> SMEFT$MB <> ", " <> tex <> "],"];
WriteLine[cfile, "  Description      -> \"Q" <> name <> " WC coefficient\""];
WriteLine[cfile, "},\n\n"];

]
(* end of WriteInternalMassTensor2Entry *)			
]



WriteInternalMassTensor4Entry = Function[{cfile,wc},
(* wc argument is member of Tensor4WC list *)
(* par file entry for 4-fermion mass basis internal WC *)
Module[{name},

name = ToString[wc[[1]]];

WriteLine[cfile, SMEFT$MB <> name <> " == {"];
WriteLine[cfile, "  ParameterType    -> Internal,"];
WriteLine[cfile, "  Indices          -> {Index[Generation], Index[Generation], Index[Generation], Index[Generation]},"];
WriteLine[cfile, "  InteractionOrder -> {NP," <> ToString[ SMEFT$NPOrder[name] ] <> "},"];
WriteLine[cfile, "  TeX              -> Superscript[" <> SMEFT$MB <> ", " <> name <> "],"];
WriteLine[cfile, "  Description      -> \"Q" <> name <> " WC coefficient\""];
WriteLine[cfile, "},\n\n"];

]
(* end of WriteInternalMassTensor4Entry *)			
]



WriteExternalScalarEntry = Function[{cfile,wc},
(* par file entry for scalar mass basis external WC *)
(* wc argument is association wcname->wcvalue(s) *)
Module[{name},

name = StringDrop[ ToString[ Keys[wc] ], 1];

WriteLine[cfile, " " <> SMEFT$MB <> name <> " == {"];
WriteLine[cfile, "    ParameterType    -> External,"];
WriteLine[cfile, "    Indices          -> {},"];
WriteLine[cfile, "    InteractionOrder -> {NP," <> ToString[ SMEFT$NPOrder[name] ] <> "},"];
WriteLine[cfile, "    Value            -> " <> NumberToString[Keys[wc] /. wc] <> "," ];
WriteLine[cfile, "    ComplexParameter -> False,"];
WriteLine[cfile, "    TeX              -> Superscript[" <> SMEFT$MB <> ", " <>
                                          StringReplace[ name, "phi"->"\[Phi]" ] <> "],"];
WriteLine[cfile, "    Description      -> \"Q" <> name <> " WC coefficient\""];
WriteLine[cfile, "    },\n\n"];

]
(* end of WriteExternalScalarEntry *)			
]



WriteExternalTensor2Entry = Function[{cfile,wc},
(* par file entry for 2-fermion mass basis external WC *)
(* wc argument is association wcname->wcvalue(s) *)
Module[{WC,tex,name},

name = ToString[ Keys[wc] ];
tex = StringReplace[name, {"phi"->"\[Phi]", "f"->"\[Phi]"} ];

WC = Values[wc];

WriteLine[cfile, " " <> SMEFT$MB <> name <> " == {"];
WriteLine[cfile, "    ParameterType    -> External,"];
WriteLine[cfile, "    Indices          -> {Index[Generation], Index[Generation]},"];
WriteLine[cfile, "    InteractionOrder -> {NP," <> ToString[ SMEFT$NPOrder[name] ] <> "},"];
WriteLine[cfile, TensorOutputFormat[ ToString /@ Keys[WC], NumberToString /@ Values[WC] ] ];
WriteLine[cfile, "    Hermitian        -> " <> If[ (Keys[wc] /. Tensor2Class) == 2, "True", "False" ] <> ","];
WriteLine[cfile, "    TeX              -> Superscript[" <> SMEFT$MB <> ", " <> tex <> "],"];
WriteLine[cfile, "    Description      -> \"Q" <> name <> " WC coefficient\""];
WriteLine[cfile, "    },\n\n"];

]
(* end of WriteExternalTensor2Entry *)			
]



WriteExternalTensor4Entry = Function[{cfile,wc},
(* par file entry for 4-fermion mass basis external WC *)
(* wc argument is association wcname->wcvalue(s) *)
Module[{name,WC},

name = ToString[ Keys[wc] ];
WC = Values[wc];

WriteLine[cfile, " " <> SMEFT$MB <> name <> " == {"];
WriteLine[cfile, "    ParameterType    -> External,"];
WriteLine[cfile, "    Indices          -> {Index[Generation], Index[Generation], Index[Generation], Index[Generation]},"];
WriteLine[cfile, "    InteractionOrder -> {NP," <> ToString[ SMEFT$NPOrder[name] ] <> "},"];
WriteLine[cfile, TensorOutputFormat[ ToString /@ Keys[WC], NumberToString /@ Values[WC] ] ];
WriteLine[cfile, "    TeX              -> Superscript[" <> SMEFT$MB <> ", " <> name <> "],"];
WriteLine[cfile, "    Description      -> \"Q" <> name <> " WC coefficient\""];
WriteLine[cfile, "    },\n\n"];

]
(* end of WriteExternalTensor4Entry *)		
]



TensorOutputFormat = Function[{wcname,wcval},
(* formats nicely output of tensor parameters for model files *)
Module[{i, tmp, sub},

sub = Table[ wcname[[i]] <> " -> " <> wcval[[i]], {i,1,Length[wcname]}];

tmp = "    Value            -> {" <> sub[[1]] <> ",\n";
For[i=2, i < Length[wcname] + 1, i++, tmp = tmp <> "                         " <> sub[[i]]  <> ",\n" ];
tmp = tmp <> "                         " <> sub[[Length[wcname]]] <> "},";

tmp

]
(* end of TensorOutputFormat *)
];



